<?php
// Minimal JWT implementation (HS256) - self-contained so Composer is optional.
function base64url_encode($data) {
  return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function base64url_decode($data) {
  $remainder = strlen($data) % 4;
  if ($remainder) $data .= str_repeat('=', 4 - $remainder);
  return base64_decode(strtr($data, '-_', '+/'));
}

function jwt_encode(array $payload, string $secret, int $exp = 3600): string {
  $header = ['alg' => 'HS256', 'typ' => 'JWT'];
  $payload['iat'] = time();
  $payload['exp'] = time() + $exp;
  $segments = [];
  $segments[] = base64url_encode(json_encode($header));
  $segments[] = base64url_encode(json_encode($payload));
  $signing_input = implode('.', $segments);
  $signature = base64url_encode(hash_hmac('sha256', $signing_input, $secret, true));
  return $signing_input . '.' . $signature;
}

function jwt_decode(string $jwt, string $secret) {
  $parts = explode('.', $jwt);
  if (count($parts) !== 3) return false;
  list($headb64, $payloadb64, $sigb64) = $parts;
  $signing_input = $headb64 . '.' . $payloadb64;
  $expected_sig = base64url_encode(hash_hmac('sha256', $signing_input, $secret, true));
  if (!hash_equals($expected_sig, $sigb64)) return false;
  $payload = json_decode(base64url_decode($payloadb64), true);
  if (!is_array($payload)) return false;
  if (isset($payload['exp']) && $payload['exp'] < time()) return false;
  return $payload;
}

?>